Rem
Rem Copyright (c) 2003, 2007, Oracle. All rights reserved.  
Rem
Rem    NAME
Rem      lcr_adm_setup.sql - SQL script to configure MGW entities.
Rem
Rem    DESCRIPTION
Rem      This script sets up a "round-trip" propagation to illustrate both
Rem      outbound (AQ->TIBRV) and inbound (TIBRV->AQ) propagation of LCRs. 
Rem      An outbound propagation job moves LCRs from the AQ queue 
Rem      MGWUSER.MGW_LCR_SRC to a user supplied TIBRV subject. An inbound 
Rem      propagation job moves the LCR from the same user supplied TIBRV 
Rem      subject to the AQ queue MGWUSER.MGW_LCR_DEST. The end result should 
Rem      be LCRs moved from MGWUSER.MGW_LCR_SRC to MGWUSER.MGW_LCR_DEST.
Rem
Rem    PREREQUISITES
Rem
Rem    * The following database entities are assumed:
Rem      - Users: mgwagent, mgwadm, mgwuser
Rem      - AQ Queues: mgwuser.mgw_lcr_src, mgwuser.mgw_lcr_dest
Rem
Rem --------------------------------------------------------------------------
Rem
Rem This script does the following (via the dbms_mgwadm package):
Rem
Rem * Sets the agent user connection information:
Rem   username: mgwagent
Rem   password: mgwagent
Rem   database: <user supplied>
Rem
Rem * Creates a TIB/Rendezvous link
Rem   Link name: tibrvlink
Rem
Rem * Creates a TIB/Rendezvous foreign queue.
Rem   Name:   destq
Rem   Link:   tibrvlink
Rem   Native: <user supplied>
Rem
Rem * Creates an OUTBOUND propagation job.
Rem   Job:    job_lcr_out
Rem   Source: mgwuser.mgw_tibrv_src
Rem   Dest:   destq@tibrvlink
Rem
Rem * Creates an INBOUND propagation job.
Rem   Job:    job_lcr_in
Rem   Source: destq@tibrvlink
Rem   Dest:   mgwuser.mgw_tibrv_dest
Rem
Rem ==========================================================================

connect mgwadm/mgwadm;

variable gv_database            VARCHAR2(32);
variable gv_tibrv_network       VARCHAR2(256);
variable gv_tibrv_service       VARCHAR2(128);
variable gv_tibrv_daemon        VARCHAR2(128);
variable gv_tibrv_subject       VARCHAR2(64);

--
-- WARNING!!!! The values below must be set properly before this script is run.
--

begin
  :gv_database         := '';     -- database specifier for agent user
  :gv_tibrv_network    := NULL;
  :gv_tibrv_service    := NULL;
  :gv_tibrv_daemon     := NULL;
  :gv_tibrv_subject    := 'oraLcr';
end;
/

prompt
prompt Setting the gateway agent user for the default agent
prompt 

rem   If we do not set the database argument it will default to the
rem   ORACLE_SID of the MGW process (the listener process).
begin
  dbms_mgwadm.alter_agent(
      agent_name => 'default_agent',
      username   => 'mgwagent',
      password   => 'mgwagent',
      database   => RTRIM(:gv_database));
end;
/

prompt
prompt Creating TIB/Rendezvous link
prompt

declare
  v_options sys.mgw_properties := NULL;
  v_prop    sys.mgw_tibrv_properties;
begin
  -- set tibrv properties used for TIB/Rendezvous connection
  v_prop := sys.mgw_tibrv_properties.construct();

  if (:gv_tibrv_network IS NOT NULL) then
    v_prop.network := RTRIM(:gv_tibrv_network);
  end if;

  if (:gv_tibrv_service IS NOT NULL) then
    v_prop.service := RTRIM(:gv_tibrv_service);
  end if;

  if (:gv_tibrv_daemon IS NOT NULL) then
    v_prop.daemon := RTRIM(:gv_tibrv_daemon);
  end if;

  -- create link
  dbms_mgwadm.create_msgsystem_link(
      linkname   => 'tibrvlink',      -- link name
      agent_name => 'default_agent',  -- MGW agent responsible for this link
      properties => v_prop,           -- tibrv link properties
      options    => v_options);       -- optional link options
end;
/

prompt
prompt Registering foreign queue
prompt

declare
  v_options sys.mgw_properties;
begin
  -- Set options for register foreign queue
  dbms_mgwadm.register_foreign_queue(
      name     => 'destq',                         -- MGW foreign queue name
      linkname => 'tibrvlink',                     -- name of link to use
      provider_queue => RTRIM(:gv_tibrv_subject),  -- TIBRV subject name
      domain   => dbms_mgwadm.DOMAIN_TOPIC,
      options  => v_options );
end;
/

prompt
prompt Adding outbound propagation job
prompt

begin
  -- source must be an AQ queue (schema.name)
  -- destination must be a registered non-Oracle queue (queue@link)
  -- subscriber rule applies to AQ subscriber
  -- exception queue, if any, must be a NORMAL AQ queue with same payload
  --   type as the source queue (schema.name)
  --
  -- for this sample:
  -- no selection rule used for this job
  -- AQ dequeue transformation: LCR (SYS.ANYDATA) to SYS.MGW_TIBRV_MSG_T
  -- no exception queue
  dbms_mgwadm.create_job(
      job_name    => 'job_lcr_out',
      propagation_type => dbms_mgwadm.outbound_propagation,
      source      => 'mgwuser.mgw_lcr_src',   -- AQ queue
      destination => 'destq@tibrvlink',       -- registered non-Oracle queue
      transformation => 'mgwadm.lcr2tibrv');
end;
/

prompt
prompt Adding inbound propagation job
prompt

begin
  -- source must be a registered non-Oracle queue (queue@link)
  -- destination must be an AQ queue (schema.name)
  -- exception queue, if any, must be a registered non-Oracle queue that
  --   uses the same link as the source queue (queue@link)
  --
  -- for this sample:
  -- no selection rule can used for this job
  -- AQ enqueue transformation: SYS.MGW_TIBRV_MSG_T to an LCR (SYS.ANYDATA)
  -- no exception queue

  dbms_mgwadm.create_job(
      job_name    => 'job_lcr_in',
      propagation_type => dbms_mgwadm.inbound_propagation,
      source      => 'destq@tibrvlink',       -- registered non-Oracle queue
      destination => 'mgwuser.mgw_lcr_dest',  -- AQ queue
      transformation => 'mgwadm.tibrv2lcr');
end;
/
